﻿///
/// アーカイブボリュームに対応するクラスのヘッダファイル。
///

#if       !defined(INCLUDED_SZ_ARCHIVE_VOLUME)
#define            INCLUDED_SZ_ARCHIVE_VOLUME

#if       !defined(INCLUDED_SZ_COMMON)
#include  <szCommon.hpp>
#endif // !defined(INCLUDED_SZ_COMMON)

#if       !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)
#include  <szForwardDeclarations.hpp>
#endif // !defined(INCLUDED_SZ_FORWARD_DECLARATIONS)

SZ_NS_BEG(szpp)

/// <summary>
/// アーカイブデータはボリュームと呼ばれる単位で複数のファイルに分割されて格納されていることがある。本クラスはそのようなボリューム一つ一つに対応する。
/// </summary>
/// <remarks>
/// 本クラスおよび派生クラスは、アーカイブボリュームのデータの実体や所在を表現するためのものであり、アーカイブの形式や格納されているファイルの情報などは保持しない。
/// そのような情報については ArchiveContent クラスを参照すること。
/// </remarks>
/// <notes>
/// アーカイブデータはストリームとして抽象化されており、通常その実体はファイルであるが、メモリ上のバッファやネットワークから取得するデータストリームであってもよい。
/// アーカイブ操作の対象は一つのアーカイブボリュームとする。つまり、たとえ圧縮データの形式が共通であっても、異なるアーカイブ間で格納アイテムのコピーや移動はできない。
/// </notes>
class SZ_SPEC ArchiveVolume
{
public:

  /// <summary>
  /// 仮想パス文字列を取得するメソッド。
  /// </summary>
  /// <remarks>
  /// アーカイブがファイルなら対応するパス文字列を持つ（定義しうる）が、アーカイブの実体が tgz を展開したオンメモリの tar データであるようなときには、対応するパス文字列が存在しない。
  /// このような場合、仮にメモリ上ではなく一時ファイルとして展開したときの仮想的なパス文字列を考える。
  /// こうすることにより、オンメモリアーカイブデータであるにもかかわらず、あたかもアーカイブファイルのように拡張子による判定などが可能になる。
  /// こうして本来はファイルでないアーカイブデータにも割り当てた仮想的なパスを仮想パス文字列と呼ぶ。
  /// </remarks>
  const szstring & GetVirtualPath() const { return DoGetVirtualPath(); }

  /// <summary>
  /// 仮想パス文字列を設定するメソッド。
  /// </summary>
  void SetVirtualPath(const szstring & path) { DoSetVirtualPath(path); }

  /// <summary>
  /// アーカイブボリュームのストリームを取得する仮想メソッド。
  /// </summary>
  /// <notes>
  /// 通常の COM ルールに従って内部で AddRef するので、呼び出し側は使い終わった時点で Release する必要がある。
  /// </notes>
  HRESULT GetArchiveStream(IInStream ** ppStream) { return DoGetArchiveStream(ppStream); }

  /// デストラクタ。
  virtual ~ArchiveVolume();

protected:

  /// 何もしないデフォルトデストラクタ。
  ArchiveVolume();

private:

  virtual const szstring & DoGetVirtualPath() const = 0;
  virtual void DoSetVirtualPath(const szstring & path) = 0;
  virtual HRESULT DoGetArchiveStream(IInStream ** ppStream) = 0;

};

SZ_NS_END(szppP)

#endif // !defined(INCLUDED_SZ_ARCHIVE_VOLUME)
